#include "NumberField.h"

#ifdef __APPLE__
#include <OpenGL/gl.h>
#else
#include <GL/gl.h>
#endif

#include <stdio.h>
#include <stdlib.h>

#include "Utilities.h"

#define NUMBER_FIELD_WIDTH 70
#define NUMBER_FIELD_HEIGHT 16

NumberField NumberField_create(bool inEditable, float inValue, float inIncrementPerPixel, void (* inValueChangeCallback)(void *, float), void * inContext, int inColor, int inPositionX, int inPositionY) {
	NumberField self;
	
	self.editable = inEditable;
	self.value = inValue;
	self.incrementPerPixel = inIncrementPerPixel;
	self.valueChangeCallback = inValueChangeCallback;
	self.context = inContext;
	self.color = inColor;
	self.positionX = inPositionX;
	self.positionY = inPositionY;
	
	return self;
}

void NumberField_setValue(NumberField * self, float value) {
	self->value = value;
	if (self->valueChangeCallback != NULL) {
		(*self->valueChangeCallback)(self->context, value);
	}
}

bool NumberField_hitTest(NumberField * self, int x, int y) {
	if (x >= self->positionX && x <= (self->positionX + NUMBER_FIELD_WIDTH) &&
	    y >= self->positionY && y <= (self->positionY + NUMBER_FIELD_HEIGHT)) {
		self->incrementMultiplier = (x + 1 - self->positionX) * 3.0f / NUMBER_FIELD_WIDTH;
		self->incrementMultiplier *= self->incrementMultiplier;
		return true;
	}
	return false;
}

void NumberField_mouseDragged(NumberField * self, int x, int y) {
	if (!self->editable) return;
	NumberField_setValue(self, self->value + (y * self->incrementPerPixel * self->incrementMultiplier));
}

void NumberField_draw(NumberField * self, bool dragging) {
	char stringValue[16];
	GLint viewport[4];
	
	glPushMatrix();
	glTranslatef(self->positionX, self->positionY, 0.0f);
	
	if (dragging || self->color != 0) {
		glBegin(GL_QUADS);
		if (dragging) {
			glColor3f(0.2f, 0.2f, 0.2f);
		} else {
			switch (self->color) {
				case 1:
					glColor3f(0.2f, 0.0f, 0.0f);
					break;
				case 2:
					glColor3f(0.0f, 0.2f, 0.0f);
					break;
				case 3:
					glColor3f(0.0f, 0.0f, 0.2f);
					break;
				case 4:
					glColor3f(0.2f, 0.2f, 0.0f);
					break;
				case 5:
					glColor3f(0.2f, 0.0f, 0.2f);
					break;
				case 6:
					glColor3f(0.0f, 0.2f, 0.2f);
					break;
			}
		}
		glVertex2i(0, 0);
		glVertex2i(NUMBER_FIELD_WIDTH, 0);
		glVertex2i(NUMBER_FIELD_WIDTH, NUMBER_FIELD_HEIGHT);
		glVertex2i(0, NUMBER_FIELD_HEIGHT);
		glEnd();
	}
	
	glBegin(GL_LINE_LOOP);
	if (dragging) {
		glColor3f(1.0f, 1.0f, 0.75f);
	} else {
		glColor3f(1.0f, 1.0f, 1.0f);
	}
	glVertex2i(0, 0);
	glVertex2i(NUMBER_FIELD_WIDTH, 0);
	glVertex2i(NUMBER_FIELD_WIDTH, NUMBER_FIELD_HEIGHT);
	glVertex2i(0, NUMBER_FIELD_HEIGHT);
	glEnd();
	
	glGetIntegerv(GL_VIEWPORT, viewport);
	glColor3f(1.0f, 1.0f, 1.0f);
	snprintf(stringValue, 9, "%.10f", self->value);
	stringValue[8] = '\x00';
	drawGlutString(3, 3, stringValue);
	
	glPopMatrix();
}
